/*
** Datei: DVIRDPK.C
** Autor: Ingo Eichenseher
**        Gerhard Wilhelms
**        Markus Zahn, 1.8.94
*/

#include <stdio.h>
#include <stdarg.h>
#include <math.h>
#include <string.h>
#ifdef UNIX
#include <sys/stat.h>
#endif
#ifdef AMIGA
#include <proto/dos.h>
#endif

#include "dvi.h"       
#include "dvimisc.h"
#if defined AMIGA || defined UNIX
#include "dvimkpk.h"
#endif

#define MF_MODE "$1"

/* Commands in the pk-files: */

#define PK_ID_BYTE 89
#define PK_XXX1 240
#define PK_XXX2 241
#define PK_XXX3 242
#define PK_XXX4 243
#define PK_YYY  244
#define PK_POST 245
#define PK_NOP  246
#define PK_PRE  247

#define VF_DEF1 243 /* define the meaning of a font number */
#define VF_DEF4 246 /* define the meaning of a font number */
#define VF_PRE  247
#define VF_ID       202
#define VF_POST 248

#define F_OK        0
#define F_WARNING   1
#define F_ERROR 2

#define GET_UNSIGNED(n,x,f) {int i_;for(x=0,i_=n;i_--;)\
    x=(x<<8l)|(getc(f)&255);}
#define GET_SIGNED(n,x,f) {int i_;for(x=(signed char)getc(f),\
    i_=n-1;i_--;)x=(x<<8l)|(unsigned char)getc(f);}


static long font_number = 0;        /* Zaehler fuer zuletzt benutzten Font */
FILE *missing = NULL;

static void fnt_tfmifree(tfm_file *tfm)
{
    if (tfm==NULL) return;
    if (tfm->lengths!=NULL)
    {
        size_t items = tfm->lengths->ec - tfm->lengths->bc + 1;
        if (tfm->width) mem_free(tfm->width,tfm->lengths->nw*sizeof(tfm_fix));
        if (tfm->finfo) mem_free(tfm->finfo,items*sizeof(tfm_finfo));
        if (tfm->header) mem_free(tfm->header,tfm->lengths->lh*sizeof(Slong));
        mem_free(tfm->lengths,sizeof(tfm_lengths));
        tfm->width = NULL;
        tfm->finfo = NULL;
        tfm->header = NULL;
        tfm->lengths = NULL;
    }
    mem_free(tfm,sizeof(tfm_file));
}

static int fnt_tfmralloc(tfm_file *tfm, void **ptr, size_t count, size_t size, FILE *fp)
{
    size_t items;
    int error = 0;

    if (count>0)
    {
        *ptr = (void*)mem_alloc(count*size,"TFM-DATA");
        items = fread(*ptr,size,count,fp);
        if (items != count)
        {
            error = 1;
            fnt_tfmifree(tfm);
            fclose(fp);
        }
    }
    return error;
}

FILE *fnt_tfmlocate(fnt_t *f)
{
    FILE *tfm;
    char n[MAX_PATH_LEN];

    sprintf(n,"%s.tfm",f->n);
    tfm = fopene(n,"rb",op.tfm_path,f->filename);
    if (tfm!=NULL) 
    {
        f->typ = f_tfm;
        f->f.tfm.tfm = NULL;
    }
    return tfm;
}

static int fnt_tfmread(fnt_t *f)
{
    FILE * fp;
    tfm_file *tfm;
    size_t items;
    unsigned i;
    long    alpha, beta;
    char *fn;

    alpha = 16*f->s; beta = 16;
    while (f->s>=0x800000l) f->s /= 2, beta /= 2;

    fn = f->filename;
    fp = fopen(fn,"rb");
    if (fp == NULL) return F_ERROR;

    f->f.tfm.tfm = NULL;
    tfm = (tfm_file*)mem_alloc(sizeof(tfm_file),"TFM-Header");
    tfm->lengths = NULL;
    tfm->header = NULL;
    tfm->finfo = NULL;
    tfm->width = NULL;

    if (fnt_tfmralloc(tfm,(void*)&tfm->lengths,1,sizeof(tfm_lengths),fp))
        return F_ERROR;
#ifdef TURN_INTS
    turn_word((char*)tfm->lengths,sizeof(tfm_lengths)/2);
#endif

    items = tfm->lengths->ec - tfm->lengths->bc + 1;
    if (fnt_tfmralloc(tfm,(void*)&tfm->header,tfm->lengths->lh,sizeof(Slong),fp))
        return F_ERROR;
#ifdef TURN_INTS
    turn_long((char*)tfm->header,tfm->lengths->lh);
#endif

    if (fnt_tfmralloc(tfm,(void*)&tfm->finfo,items,sizeof(tfm_finfo),fp))
        return F_ERROR;

    if (fnt_tfmralloc(tfm,(void*)&tfm->width,tfm->lengths->nw,sizeof(tfm_fix),fp))
        return F_ERROR;

    f->f.tfm.tfm = tfm;

    for (i=0; i<tfm->lengths->nw; i++)
    {
        char b0 = tfm->width[i].b[0];
        tfm->width[i].l = (((((tfm->width[i].b[3]*f->s)/256l)+
                            (tfm->width[i].b[2]*f->s))/256l)+
                           (tfm->width[i].b[1]*f->s))/beta;
        if (b0) tfm->width[i].l -= alpha;
    }

    if (f->c && f->c!=tfm->header->check_sum && f->warn)
        print("Warning: TFM-File %s has wrong checksum",fn);

    if (f->d != (tfm->header->design_size>>4) && f->warn)
        print("Warning: TFM-File %s has wrong designsize",fn);

    f->warn = 0;
    return F_OK;
}

static void fnt_tfmfree(fnt_t *f)
{
    fnt_tfmifree(f->f.tfm.tfm);
    f->f.tfm.tfm = NULL;
}

#ifdef PK_FULLCONFIG

void fnt_pkname(        char *name, char *schab, char *fn, int isize, int rsize )
{
    /*
     * M.Z., 27.10.1992
     * Erzeugt den in der Schablone 'schab' definierten Fontnamen in 'name'.
     * Die Schablone kann folgende Kontrollzeichen enthalten:
     * - %d: Fuegt die dpi-Zahl des Fonts ein
     * - %h: Fuegt die horizontale Basis-dpi-Aufloesung ein
     * - %m: Fuegt "Magnification" analog zum Atari-"Lindner"-TeX ein
     * - %s: Fuegt den Fontnamen ein
     * - %v: Fuegt die vertikale Basis-dpi-Aufloesung ein
     * - %%: Fuegt schliesslich ein "%" ein
     * => "%s.%dpk" ergibt dann z.B. Namen wie "cmr10.300pk"
     * => "RES300.PS\MAG%m\%s.pk" liefert dann z.B. Namen wie 
     *    "RES300.PS\MAG____1.200\cmr10.pk"
     */
    char *p, *q;

    *name = '\0';
    for( p = schab; *p; p++ )
        if( *p == '%' )
        {
            switch( *(++p) )
            {
                case 'd':
                    sprintf( name + strlen( name ), "%d", rsize );
                    break;
                case 'h':
                    sprintf( name + strlen( name ), "%ld", op.hres );
                    break;
                case 'm':
                    sprintf( name + strlen( name ), "%5d.%03d", isize/1000,
                             isize%1000 );
                    for( q = name; *q != '\0'; q++ )
                        if (*q==' ') *q='_';
                    break;
                case 's':
                    sprintf( name + strlen( name ), "%s", fn );
                    break;
                case 'v':
                    sprintf( name + strlen( name ), "%d", op.vres );
                    break;
                case '%':
                    strcat( name, "%" );
                    break;
                default:
                    print( "Illegal argument %c in pkname: %s", *p, schab );
                    break;
            }
        }
        else
            sprintf( name + strlen( name ), "%c", *p );
}

#endif

static FILE *fnt_pktry(char *fn, int isize, int rsize, char *n)
{
    FILE *pk = NULL;
    char path[MAX_PATH_LEN];
    char *q, *p = op.pk_path;

    while(*p!='\0' && pk==NULL)
    {
        for(q=path; *p && *p!=PATH_SEP; *q++ = *p++);
        if (*p==PATH_SEP) p++;
        *q = '\0';

#ifndef PK_FULLCONFIG
        if (strchr(path,'%'))
        {
            sprintf(n,path,rsize);
            sprintf(n+strlen(n),"%c%s.pk",BSLASH,fn);
        }
        else
        {
            char *lp = fname(path);
            if (strchr(lp,'.')!=NULL && strcmp(lp,".") && strcmp(lp,".."))
            {
                sprintf(n,"%s%cmag%5d.%03d%c%s.pk",path,BSLASH,
                    isize/1000,isize%1000,BSLASH,fn);
                for(q=n; *q!='\0'; q++) if (*q==' ') *q='_';
            }
            else sprintf(n,"%s%c%s.%dpk",path,BSLASH,fn,rsize);
        }
#else
        fnt_pkname( n, path, fn, isize, rsize );
#endif
        pk = fopen(n,"rb");

        if( pk == NULL && op.showfonts )
            print( "PK : %s failed", n );
    }

    return pk;
}

FILE *fnt_pklocate(fnt_t *f)
{
    FILE *pk;
    double size = (double)f->s*dvi_info.mag/1e3/(double)f->d;
    int isize = iround(size*1000.0);
    int rsize = iround(size*(double)op.hres);

    pk = fnt_pktry(f->n,isize,rsize,f->filename);
    if (pk!=NULL) f->typ = f_pk;
    else if ((pk = fnt_pktry(f->n,isize-1,rsize-1,f->filename))!=NULL) f->typ = f_pk;
    else if ((pk = fnt_pktry(f->n,isize+1,rsize+1,f->filename))!=NULL) f->typ = f_pk;
    if (f->typ==f_pk)
    {
        f->f.pk.defined = NULL;
        f->f.pk.hash = NULL;
    }
    return pk;
}

static int fnt_pkread(fnt_t *f)
{
    int     i, k, flag_byte, err = F_OK;
    pk_char c, *c1;
    long    j, dm;
    long    z, alpha, beta;
    int     b0, b1, b2, b3;
    FILE    *pk = NULL;
    char    *fn;
    double  hfres, vfres;

    fn = f->filename;
    pk = fopen(fn,"rb");

    if (pk==NULL) return F_ERROR;

    z = f->s;
    alpha = 16*z; beta = 16;
    while (z>=0x800000l) z/=2,beta/=2;

    f->f.pk.hash = (pk_char**)mem_alloc(HASHSIZE*sizeof(pk_char*),"PK-Hashtable");
    for (i=0; i<HASHSIZE; i++) f->f.pk.hash[i]=NULL;

    if (getc(pk)!=PK_PRE)
    {
        print("Error: PK-file %s does not start with PRE",fn);
        fclose(pk);
        return F_ERROR;
    }
    if (getc(pk)!=PK_ID_BYTE)
    {
        print("Error: PK-file %s has wrong ID-byte",fn);
        fclose(pk);
        return F_ERROR;
    }

    for (k=getc(pk); k--;) (void)getc(pk);
    GET_UNSIGNED(4,f->design_size,pk);
    GET_UNSIGNED(4,f->check_sum,pk);
    GET_UNSIGNED(4,f->f.pk.hppp,pk);
    GET_UNSIGNED(4,f->f.pk.vppp,pk);

    hfres = f->f.pk.hppp/65.536*72.27*f->d/(double)f->s/(double)dvi_info.mag;
    vfres = f->f.pk.vppp/65.536*72.27*f->d/(double)f->s/(double)dvi_info.mag;

    if (f->c && f->c!=f->check_sum)
    {   
        if (f->warn) print("Warning: PK-file %s has a wrong checksum",fn); 
        err = F_WARNING;
    }
    if ( f->d!= (f->design_size>>4) )
    {   
        if (f->warn) print("Warning: PK-file %s wrong designsize",fn);
        err = F_WARNING;
    }

    if ( lround(hfres)!=op.hres )
    {   
        if (f->warn) print("Warning: PK-file %s wrong h-resolution %f",fn,hfres); 
        err = F_WARNING;
    }
    if ( lround(vfres)!=op.vres )
    {   
        if (f->warn) print("Warning: PK-file %s wrong v-resolution %f",fn,vfres); 
        err = F_WARNING;
    }
    f->warn = 0;

    while( (flag_byte=getc(pk))!=PK_POST )
    {
        switch(flag_byte)
        {
            case PK_XXX1:
            case PK_XXX2:
            case PK_XXX3:
            case PK_XXX4:
                GET_UNSIGNED(flag_byte-PK_XXX1+1,j,pk);
                while(j--) (void)getc(pk);
                break;
            case PK_YYY:
                GET_UNSIGNED(4,j,pk);
                break;
            case PK_NOP:
                break;
            case PK_PRE: case PK_POST:
            case 248: case 249: case 250: case 251:
            case 252: case 253: case 254: case 255:
                print("Error: Illegal command in PK-file %s",fn);
                fclose(pk);
                return F_ERROR;
            default:
                c.dyn_f = flag_byte>>4;
                c.state = (flag_byte & 8)!=0;
                switch(flag_byte&7)
                {
                    case 0:
                    case 1:
                    case 2:
                    case 3:
                        GET_UNSIGNED(1,c.pl,pk); c.pl += ((flag_byte&3)<<8);
                        GET_UNSIGNED(1,c.cc,pk);
                        b0=0;
                        b1=getc(pk);
                        b2=getc(pk);
                        b3=getc(pk);
                        GET_UNSIGNED(1,dm,pk);
                        GET_UNSIGNED(1,c.w,pk);
                        GET_UNSIGNED(1,c.h,pk);
                        GET_SIGNED(1,c.hoff,pk);
                        GET_SIGNED(1,c.voff,pk);
                        c.dx = dm<<16l; c.dy=0;
                        c.pl -= 8;
                        break;
                    case 4:
                    case 5:
                    case 6:
                        GET_UNSIGNED(2,c.pl,pk); c.pl += ((flag_byte&3)<<16);
                        GET_UNSIGNED(1,c.cc,pk);
                        b0=0;
                        b1=getc(pk);
                        b2=getc(pk);
                        b3=getc(pk);
                        GET_UNSIGNED(2,dm,pk);
                        GET_UNSIGNED(2,c.w,pk);
                        GET_UNSIGNED(2,c.h,pk);
                        GET_SIGNED(2,c.hoff,pk);
                        GET_SIGNED(2,c.voff,pk);
                        c.dx = dm<<16l; c.dy=0;
                        c.pl -= 13;
                        break;
                    case 7:
                        GET_UNSIGNED(4,c.pl,pk);
                        GET_UNSIGNED(4,c.cc,pk);
                        b0=getc(pk);
                        b1=getc(pk);
                        b2=getc(pk);
                        b3=getc(pk);
                        GET_UNSIGNED(4,c.dx,pk);
                        GET_UNSIGNED(4,c.dy,pk);
                        GET_UNSIGNED(4,c.w,pk);
                        GET_UNSIGNED(4,c.h,pk);
                        GET_SIGNED(4,c.hoff,pk);
                        GET_SIGNED(4,c.voff,pk);
                        c.pl -= 28;
                        break;
                }
                c.tfm=(((((b3*z)/256l)+(b2*z))/256l)+(b1*z))/beta;
                if (b0) c.tfm -= alpha;

                c1 = (pk_char*)mem_alloc(sizeof(pk_char)+c.pl,"PK-Char-Data");
                *c1 = c;
                c1->next = f->f.pk.hash[(int)c.cc%HASHSIZE];
                f->f.pk.hash[(int)c.cc%HASHSIZE] = c1;

                if (c.pl!=fread(c1->data,1,(size_t)c.pl,pk))
                {
                    fclose(pk);
                    print("Error reading PK-File %s",fn);
                    return F_ERROR;
                }
                break;
        }
    }
    fclose(pk);

    return err;
}

static void fnt_pkfree(fnt_t *f, int total)
{
    pk_char *p, *q;
    int i;

    if (f->f.pk.hash!=NULL)
    {
        for (i=0; i<HASHSIZE; i++)
            for (p=f->f.pk.hash[i]; p!=NULL; p=q)
            {
                q = p->next;
                mem_free(p,sizeof(pk_char)+p->pl);
            }
        mem_free(f->f.pk.hash,HASHSIZE*sizeof(pk_char*));
        f->f.pk.hash = NULL;
    }

    if (total)
    {
        def_list *p, *q;

        p = f->f.pk.defined;
        f->f.pk.defined = NULL;

        while(p!=NULL)
        {
            q = p->next;
            mem_free(p,sizeof(def_list));
            p = q;
        }
    }
}

static void fnt_notfound(double size, char *n)
{
    static int first_time = 1;
    char pathname[64], *q;
    int magstep;
    int isize = iround(size*1000.0);
    double dmagstep;

    dmagstep = log(size)/log(1.2);
/*    printf("\nSize: %lf, isize: %d, dmag: %lf\n", size, isize, dmagstep);*/
    if ( fabs( dmagstep - 0.5 ) < 5e-3 )
    {
       size = sqrt( 1.2 );
/*       printf("Rounded size to %lf\n", size);*/
    }

    magstep = iround( dmagstep );
/*    printf("magstep: %d\n", magstep );*/
    if (fabs(exp(magstep*log(1.2))-size)>5e-5)
    {
      magstep = -1;
/*      printf("Not rounded\n");*/
    }
    else
    {
      size = exp(magstep*log(1.2));
/*      printf("Rounded to %lf\n", size);*/
    }
/*    printf("Size: %lf\n", size);*/

    if (missing==NULL)
    {
            missing     = fopen(MISSING,first_time ? "w":"a");
            if (missing==NULL)
                halt("Cannot open file %s for output",MISSING);
#if defined(IBMPC) && !defined(UNIAUG) /* IBMPC and others */
            fputs("@IF \"%1\"==\"\"     GOTO stop\n",missing);
            fputs("@IF NOT EXIST %1\\nul mkdir %1\n",missing);
#endif
#ifdef AMIGA
        if( first_time )
        {
            fputs( ".key mode,final/f\n", missing );
            fputs( ".bra {\n", missing );
            fputs( ".ket }\n", missing );
        }
#endif
            first_time = 0;
    }

    sprintf(pathname,"mag%5d.%03d",isize/1000,isize%1000);
    for( q = pathname; *q!='\0'; q++ )
        if ( *q == ' ' )
                *q = '_';

    if ( n != NULL )
    {

#ifdef UNIX
        if (magstep!=-1)
            fprintf(missing,
             "mf \"\\mode:=%s; nonstopmode; mag:=magstep%d; input %s\"\n",
             MF_MODE,magstep,n);
        else
            fprintf(missing,
             "mf \"\\mode:=%s; nonstopmode; mag:=%.10f; input %s\"\n",
             MF_MODE,size,n);
        fprintf(missing,
            "gftopk %s.%dgf %s.%dpk\n",n,iround(size*(double)op.hres),n,
            iround(size*(double)op.hres));
#endif

#ifdef ATARI_ST
        char pname[128];
#ifdef PK_FULLCONFIG
        char *respointer;
#endif

#ifndef PK_FULLCONFIG
        getfile(pname,op.pk_path);
#else
/*
 * jetzt muss der RESxxx-Teil in op.pk_path gefunden und
 * expandiert werden
 */
        respointer = strstr( op.pk_path, "RES" );
        if ( respointer == NULL )
            respointer = strstr( op.pk_path, "res" );
/* wer Klein- und Grossschreibung mischt, den bestraft das Leben */
        if ( respointer != NULL )
        {
            fnt_pkname( pname, respointer, n, isize,
                iround( size * (double)op.hres ) );
            respointer = pname;
            while ( *respointer != BSLASH && *respointer != '\0' )
                respointer++;
            *respointer = '\0';
        }
#endif
        strlwr(pname);
        if (magstep!= -1)
            fprintf(missing,"%s %s magstep%d\n",pname,n,magstep);
        else
            fprintf(missing,"%s %s %.10lf\n",pname,n,size);
#endif

#if defined(IBMPC) && !defined(UNIAUG) /* IBMPC and others */
        if (magstep!=-1)
            fprintf(missing,
             "mf \"\\mode:=%%1; nonstopmode; mag:=magstep%d; input %s\"\n",
             magstep,n);
        else
            fprintf(missing,
             "mf \"\\mode:=%%1; nonstopmode; mag:=%.10f; input %s\"\n",
             size,n);
        fprintf(missing,"@IF NOT EXIST %%1\\%s\\nul mkdir %%1\\%s\n",pathname,pathname);
        fprintf(missing,
            "gftopk %s.%dgf %%1\\%s\\%s.pk\n",n,iround(size*(double)op.hres),
            pathname,n);
        fprintf(missing,"@del %s.%dgf\n",n,iround(size*(double)op.hres));
#endif

#if defined(IBMPC) && defined(UNIAUG)
        fprintf(missing,
            "command /E:1648 /C makegf %s plain %ld %.10lf\n",
            n, op.hres, size );
        fprintf(missing,
            "command /E:1648 /C makepk %s %ld %d\n",
            n, op.hres, iround( size*(double)op.hres ) );
#endif

#ifdef AMIGA
    char pkname[MAX_PATH_LEN];

    fnt_pkname( pkname, op.pk_path, n, isize, iround( size * (double)op.hres ) );

    fprintf( missing, "\nif exists %s\n", pkname );
    fprintf( missing, "  echo \"%s already exists!\"\n", pkname );
    fputs( "else\n", missing );
    fputs( "  cd >ENV:dvidir{$$}\n", missing );
    fputs( "  cd T:\n", missing );

    if( magstep == -1 )
        fprintf( missing,
            "  virmf \"\\mode:={mode}; nonstopmode; mag:=%.10f; input %s\"\n",
            size, n );
    else
        fprintf( missing,
            "  virmf \"\\mode:={mode}; nonstopmode; mag:=magstep(%d); input %s\"\n",
            magstep, n );
        fprintf(missing, "  gftopk %s.%dgf %s\n", n, iround(size*(double)op.hres),
        pkname );

    fprintf( missing, "  delete %s#?\n", n );
    fputs( "  cd \"$dvidir{$$}\"\n", missing );
        fputs( "endif\n", missing );
#endif

    }
}

static FILE *fnt_vflocate(fnt_t *f)
{
    FILE *vf;
    char n[MAX_PATH_LEN];
    sprintf(n,"%s.vf",f->n);
    vf = fopene(n,"rb",op.vf_path,f->filename);
    if (vf!=NULL) 
    {
        f->typ = f_vf;
        f->f.vf.hash = NULL;
        f->f.vf.fl = NULL;
    }
    return vf;
}

static int fnt_vfread(fnt_t *f)
{
    int k, c, err = F_OK;
    FILE *vf;
    char *fn;
    long alpha, beta;
    int b0, b1, b2, b3;

    alpha = 16*f->s; beta = 16;
    while (f->s>=0x800000l) f->s /= 2, beta /= 2;

    fn = f->filename;
    vf = fopen(fn,"rb");
    if (vf==NULL) return F_ERROR;

    f->f.vf.hash = (vf_char**)mem_alloc(HASHSIZE*sizeof(vf_char*),"VF-Hashtable");
    for (k=0; k<HASHSIZE; k++) f->f.vf.hash[k] = NULL;
    f->f.vf.fl = NULL;

    if (getc(vf)!=VF_PRE)
    {
        fclose(vf);
        halt("Illegal preamble in vf file %s",fn);
    }
    if (getc(vf)!=VF_ID)
    {
        fclose(vf);
        halt("Illegal id in vf file %s",fn);
    }

    for (k=getc(vf); k--;) (void)getc(vf);
    GET_UNSIGNED(4,f->check_sum,vf);
    GET_UNSIGNED(4,f->design_size,vf);

    if (f->c && f->check_sum!=f->c)
    {
        if (f->warn) print("Warning: VF-File %s has a wrong checksum",fn);
        err = F_WARNING;
    }
    if ((f->design_size>>4)!=f->d)
    {
        if (f->warn) print("Warning: VF-File %s has a wrong designsize",fn);
        err = F_WARNING;
    }

    f->warn = 0;

    while((c=getc(vf))>=VF_DEF1 && c<=VF_DEF4)
    {
        long k, cs, s, d, a, l;
        char n[MAX_PATH_LEN];

        GET_UNSIGNED(c-VF_DEF1+1,k,vf);
        GET_UNSIGNED(4,cs,vf);
        GET_UNSIGNED(4,s,vf);
        GET_UNSIGNED(4,d,vf);
        a = getc(vf);
        l = getc(vf);
        if (a+l>MAX_PATH_LEN) halt("Fontname too long");
        fread(n,1,(size_t)(a+l),vf);
        n[(int)(a+l)] = '\0';
        d >>= 4;
        s = (long)(d*(s/1048576.0)*((double)f->s/(double)f->d));

        (void)fnt_define(&f->f.vf.fl,k,cs,s,d,n);
    }

    for(; c<243; c=getc(vf))
    {
        long pl, cc;
        vf_char *v;

        if (c==242)
        {
            GET_UNSIGNED(4,pl,vf);
            GET_UNSIGNED(4,cc,vf);
            b0 = getc(vf);
            b1 = getc(vf);
            b2 = getc(vf);
            b3 = getc(vf);
        }
        else
        {
            pl = c;
            GET_UNSIGNED(1,cc,vf);
            b0 = 0;
            b1 = getc(vf);
            b2 = getc(vf);
            b3 = getc(vf);
        }

        v = (vf_char*)mem_alloc(sizeof(*v)+pl,"VF-Data");
        v->pl = pl;
        v->cc = cc;
        v->tfm = (((((b3*f->s)/256l) + (b2*f->s))/256l) + (b1*f->s))/beta;
        if (b0) v->tfm -= alpha;

        v->next = f->f.vf.hash[(int)cc%HASHSIZE];
        f->f.vf.hash[(int)cc%HASHSIZE] = v;

        if (v->pl!=fread(v->data,1,(size_t)v->pl,vf))
        {
            fclose(vf);
            print("Error reading VF-File %s",fn);
            return F_ERROR;
        }
    }

    if (c!=VF_POST)
    {
        fclose(vf);
        halt("Illegal postamble in vf file %s",fn);
    }

    fclose(vf);

    return err;
}

static void fnt_vffree(fnt_t *f, int total)
{
    vf_char *p, *q;
    int i;

    if (f->f.vf.hash!=NULL)
    {
        for (i=0; i<HASHSIZE; i++)
            for (p=f->f.vf.hash[i]; p!=NULL; p=q)
            {
                q = p->next;
                mem_free(p,sizeof(vf_char)+p->pl);
            }
        mem_free(f->f.vf.hash,HASHSIZE*sizeof(vf_char*));
        f->f.vf.hash = NULL;
    }

    if (total) 
    {
        fnt_lfree(f->f.vf.fl);
        f->f.vf.fl = NULL;
    }
}

void fnt_select(fnt_t *f)
{
    if (!f->loaded)
    {
        if (op.showfonts) xprint(0,"(%s)",f->n);
        switch(f->typ)
        {
            case f_pk: fnt_pkread(f); break;
            case f_vf: fnt_vfread(f); break;
            case f_tfm: fnt_tfmread(f); break;
            case f_ps: fnt_tfmread(f); break;
        }
    }
    f->loaded = 1;
    f->number = font_number++;
}


int fnt_define(fnt_t **fl, long k, long c, long s, long d, char *n)
{
    register fnt_t  **f;
    int                 ok = 0;
    double              size = (double)s*dvi_info.mag/1e3/(double)d;
    FILE                *fp;

    for(f=fl; *f!=NULL; f=&(*f)->next) if ((*f)->k==k) return 1;

    *f = (fnt_t*)mem_alloc(sizeof(fnt_t),"Font-Header");
    (*f)->n = mem_strdup(n);

    (*f)->k = k;            /* Font Identifier */
    (*f)->c = c;            /* checksum */
    (*f)->s = s;            /* scaled size */
    (*f)->d = d;            /* design size */
    (*f)->next = NULL;      /* Nachfolger in der Font-Liste */
    (*f)->space = s/6;      /* Font normal space */
    (*f)->typ = f_not_found;/* noch nicht gefunden */
    (*f)->warn = 1;
    (*f)->loaded = 0;
    (*f)->locked = 0;
    (*f)->number = 0;
    (*f)->ps_name = 0;

    strcpy((*f)->n,n);

    if (resident(*f))
    {
        print("RES: %s",(*f)->n);
        return 1;
    }

    if ((fp = fnt_pklocate(*f))!=NULL)
    {
        fclose(fp);
        psfont_def(*f,size);
        ok = 1;
        if (op.showfonts) print("PK : %s",(*f)->filename);
    }

    if (!ok && (fp=fnt_vflocate(*f))!=NULL)
    {
        fclose(fp);
        ok = 1;
        if (op.showfonts) print("VF : %s",(*f)->filename);
    }

#if defined (AMIGA) || defined (UNIX)

    /*
     * M.Z., 09.09.1992
     * Falls Font noch nicht gefunden wird er evtl. ueber ein
     * externes ARexx-Script generiert und dann geladen.
     */

    if( !ok && ( ( fp = fnt_mkpk( *f ) ) != NULL ) )
    {
            fclose(fp);
            psfont_def(*f,size);
            ok = 1;
            if (op.showfonts) print("PK : %s",(*f)->filename);
    }

#endif

    if (!ok) fnt_notfound(size,n);

    if (!ok)
    {
        if ((fp=fnt_tfmlocate(*f))!=NULL)
        {
            fclose(fp);
            if (op.showfonts) print("TFM: %s",(*f)->filename);
            print("Warning: Font %s (%d) only as TFM-File",(*f)->n,iround(size*1000));
        }
        else print("Warning: Found no PK, VF or TFM for Font %s (%d)",n,iround(size*1000));
    }

    return ok; 
}

void fnt_free(fnt_t *f, int total)
{
    f->loaded = 0;
    switch(f->typ)
    {
        case f_pk: fnt_pkfree(f,total); break;
        case f_vf: fnt_vffree(f,total); break;
        case f_ps: 
        case f_tfm: fnt_tfmfree(f); break;
    }
    if (total)
    {
        if (f->ps_name!=NULL) mem_strfree(f->ps_name);
        if (f->n) mem_strfree(f->n);
        f->n = NULL;
        f->ps_name = NULL;
    }
    else if (op.showfonts) xprint(0,"(-%s)",f->n);
}

fnt_t *fnt_find(fnt_t *fl)
{
    fnt_t *f, *best = NULL;
    long number = font_number;

    for(f = fl; f!=NULL; f=f->next)
    {
        if (f->loaded && f->number<number && !f->locked)
        {
            number = f->number;
            best = f;
        }
        if (f->typ==f_vf)
        {
            fnt_t *g = fnt_find(f->f.vf.fl);
            if (g!=NULL && g->number<number) best = g;
        }
    }
    return best;
}

void fnt_lfree(fnt_t *f)
{
    fnt_t  *q;

    while(f!=NULL)
    {
        q = f->next;
        fnt_free(f,1);
        mem_free(f,sizeof(fnt_t));
        f = q;
    }
}

void close_missing(void)
{
    if (missing!=NULL)
    {
#ifdef IBMPC
        fputs(":stop\n",missing);
#endif
        fclose(missing);
        /* Jetzt: missing Datei ausf"uhrbar machen */
#ifdef UNIX     
        chmod( MISSING, 0755 );
#endif  
#ifdef AMIGA
        SetProtection( MISSING, FIBF_SCRIPT | FIBF_EXECUTE );
#endif
        missing = NULL;
    }
}

